#include "line.h"

/*!
  \file
  \brief Line functions
*/


#ifndef DOXYGEN_SHOULD_SKIP_THIS

  PG_FUNCTION_INFO_V1(sphereline_in);
  PG_FUNCTION_INFO_V1(sphereline_from_point);
  PG_FUNCTION_INFO_V1(sphereline_from_points);
  PG_FUNCTION_INFO_V1(sphereline_from_trans);
  PG_FUNCTION_INFO_V1(sphereline_meridian);
  PG_FUNCTION_INFO_V1(sphereline_swap_beg_end);
  PG_FUNCTION_INFO_V1(sphereline_begin);
  PG_FUNCTION_INFO_V1(sphereline_end);
  PG_FUNCTION_INFO_V1(sphereline_length);
  PG_FUNCTION_INFO_V1(sphereline_cont_point);
  PG_FUNCTION_INFO_V1(sphereline_cont_point_neg);
  PG_FUNCTION_INFO_V1(sphereline_cont_point_com);
  PG_FUNCTION_INFO_V1(sphereline_cont_point_com_neg);
  PG_FUNCTION_INFO_V1(spherecircle_cont_line);
  PG_FUNCTION_INFO_V1(spherecircle_cont_line_neg);
  PG_FUNCTION_INFO_V1(spherecircle_cont_line_com);
  PG_FUNCTION_INFO_V1(spherecircle_cont_line_com_neg);
  PG_FUNCTION_INFO_V1(sphereline_overlap_circle);
  PG_FUNCTION_INFO_V1(sphereline_overlap_circle_neg);
  PG_FUNCTION_INFO_V1(sphereline_overlap_circle_com);
  PG_FUNCTION_INFO_V1(sphereline_overlap_circle_com_neg);
  PG_FUNCTION_INFO_V1(sphereline_equal);
  PG_FUNCTION_INFO_V1(sphereline_equal_neg);
  PG_FUNCTION_INFO_V1(sphereline_crosses);
  PG_FUNCTION_INFO_V1(sphereline_crosses_neg);
  PG_FUNCTION_INFO_V1(sphereline_overlap);
  PG_FUNCTION_INFO_V1(sphereline_overlap_neg);
  PG_FUNCTION_INFO_V1(spheretrans_from_line);
  PG_FUNCTION_INFO_V1(spheretrans_line);
  PG_FUNCTION_INFO_V1(spheretrans_line_inverse);

#endif

  /*!
    \brief returns a point at line at given length position 
    \param len  length position
    \param l    pointer to line
    \param p    pointer to spherical point
    \return true, if point at line
  */
  static bool sline_point_by_length ( SPoint * p , const SLine * l, float8 len )
  {
    
    static SEuler se ;
    static SPoint sp = { 0.0, 0.0 };
    if ( 0.0 >  len || len > l->length ) return false; 

    seuler_set_zxz ( &se );
    se.phi   = l->phi;
    se.theta = l->theta;
    se.psi   = l->psi;

    sp.lng   = len  ;
    euler_spoint_trans( p, &sp, &se );
    return true;
  } 


  /*!
    \brief Swaps begin and end of a line
    \param out the result pointer to a spherical line
    \param in  the input pointer to a spherical line
    \return a pointer to swapped spherical line 
  */
  static SLine * sline_swap_beg_end ( SLine * out, const SLine * in )
  {
    static SLine   l;
    static SEuler se;
    l.length =   in->length;
    l.phi    = - in->length;
    l.theta  = PI ;
    l.psi    = 0.0;
    seuler_set_zxz( &se );
    se.phi   = in->phi;
    se.theta = in->theta;
    se.psi   = in->psi;
    euler_sline_trans ( out , &l , &se );
    return out;
  }


  bool sline_eq ( const SLine * l1, const SLine * l2 )
  {

    if ( FPne( l1->length, l2->length) ){
      return false;
    } else {
      static SEuler e1, e2;
      seuler_set_zxz ( &e1 );
      seuler_set_zxz ( &e2 );
      e1.phi   = l1->phi;
      e1.theta = l1->theta;
      e1.psi   = l1->psi;
      e2.phi   = (FPeq(l2->length,PID))?(l1->phi):(l2->phi);
      e2.theta = l2->theta;
      e2.psi   = l2->psi;
      return ( strans_eq ( &e1, &e2 ) );
    }
    
    return FALSE;

  }


  bool sline_from_points ( SLine * sl, const SPoint * pbeg, const SPoint * pend )
  {

    static SEuler  se;
    static float8   l;

    l = spoint_dist ( pbeg, pend );

    if ( FPeq ( l,PI ) ){
      if ( FPeq( pbeg->lng, pend->lng ) ){
        sline_meridian( sl, pbeg->lng );
        return TRUE;
      }
      return FALSE;
    }

    if ( spherevector_to_euler ( &se, pbeg, pend ) ){
      sl->phi    = se.phi;
      sl->theta  = se.theta;
      sl->psi    = se.psi;
      sl->length = l;
    } else {
      sl->phi    = PIH ;
      sl->theta  = pbeg->lat ;
      sl->psi    = pbeg->lng - PIH ;
      sl->length = 0.0;
    }
    return ( TRUE );
    
  }

  SLine * sline_meridian( SLine * sl, float8 lng )
  {
    static SPoint p;
    sl->phi    = - PIH;
    sl->theta  =   PIH;
    p.lat      =   0.0;
    p.lng      =   lng;
    spoint_check(&p);
    sl->psi    = p.lng;
    sl->length = PI;
    return sl;
  } 


  SPoint * sline_begin ( SPoint * p, const SLine  * l )
  {
    static SPoint  tmp = { 0.0, 0.0 } ;
    static SEuler   se ;
    sphereline_to_euler ( &se, l );
    euler_spoint_trans ( p, &tmp, &se );
    return p;
  }

  SPoint * sline_end ( SPoint * p, const SLine  * l )
  {
    static SPoint  tmp  = { 0.0, 0.0 } ;
    static SEuler   se  ;
    tmp.lng = l->length ;
    sphereline_to_euler ( &se, l );
    euler_spoint_trans ( p, &tmp, &se );
    return p;
  }


  void sline_min_max_lat ( const SLine * sl , float8 * minlat, float8 * maxlat )
  {

    float8 inc = sl->theta - floor ( sl->theta / PI ) * PI ;
    

    if ( FPzero ( inc ) || FPeq ( inc, PI ) )
    {

    	*minlat = *maxlat = 0.0;
    	return;

    } else {

      SEuler se ;
      SLine  nl ;
      SPoint tp ;

      float8 lng ;

      seuler_set_zxz( &se );
      se.phi   = -sl->psi ;
      se.theta = ( inc > PIH )?( PI-2*inc ):( 0.0 ) ;
      se.psi   = 0.0;
      euler_sline_trans ( &nl , sl , &se );

      // Now ascending node at (0,0) , line ascending

      sline_begin ( &tp , &nl );
      *minlat = *maxlat =  tp.lat;

      sline_end   ( &tp , &nl );
      *minlat = min ( tp.lat, *minlat );
      *maxlat = max ( tp.lat, *maxlat );

      for ( lng = PIH ; lng < PID  ; lng += PI )
      {
        tp.lng = lng;
        tp.lat = ( lng < PI ) ? ( inc ) : ( -inc ) ;
        if ( spoint_at_sline( &tp, &nl ) )
        {
          *minlat = min ( tp.lat, *minlat );
          *maxlat = max ( tp.lat, *maxlat );
        }
      }
    
    }

  }



  int8 sphereline_circle_pos ( const SLine * sl , const SCIRCLE * sc )
  {

    static float8 i, mi;
    static const float8 step = ( PI - 0.01 );
    static SPoint p[2] = { { 0.0, 0.0} , { 0.0, 0.0 } } ;
    static SCIRCLE  c ;
    static bool   bbeg, bend ;
    static SEuler se ;
    static int    contain;

    if ( FPzero( sl->length ) ) { // line is point
      sline_begin( &p[0], sl ) ;
      if ( spoint_in_circle ( &p[0], &c ) ){
        return PGS_CIRCLE_CONT_LINE  ;
      } else {
        return PGS_CIRCLE_LINE_AVOID ;
      }
    }

    contain = 0;
    sphereline_to_euler_inv ( &se, sl );
    euler_scircle_trans ( &c, sc, &se );

    mi = sl->length / step ;

    // split line in segments and check for each of it

    for ( i=0.0; i<mi; i += 1.0 ){

      p[0].lng = i * step ;
      p[1].lng = ( ( ( i+1.0 ) > mi )?( sl->length ):( ( i+1.0 ) * step ) ) ;

      bbeg     = spoint_in_circle ( &p[0], &c );
      bend     = spoint_in_circle ( &p[1], &c );

      if ( bbeg  && bend   ){
        contain++;
      } else
      if ( bbeg  || bend   ){
        return PGS_CIRCLE_LINE_OVER;
      } else
      if (
           FPle( ((c.center.lat<0)?(-c.center.lat):(c.center.lat)), c.radius ) &&
           FPge( c.center.lng, p[0].lng ) && FPle(c.center.lng, p[1].lng )
         )
      {
        return PGS_CIRCLE_LINE_OVER ;
      } else
      if ( contain > 0 ){
        return PGS_CIRCLE_LINE_OVER ;
      }
    }

    if ( contain > 0 && contain == ( floor (mi) + 1 ) ){
      return PGS_CIRCLE_CONT_LINE ;
    }

    return PGS_CIRCLE_LINE_AVOID ;

  }


  bool sline_circle_touch( const SLine * sl , const SCIRCLE * sc )
  {
    // we assume here, line and circle are overlapping
    static SEuler se;
    static SCIRCLE tc;

    sphereline_to_euler_inv ( &se, sl );
    euler_scircle_trans  ( &tc , sc , &se );
    if ( FPge(tc.center.lng,0.0) && FPle(tc.center.lng,sl->length) ){
      if ( FPeq( fabs(tc.center.lat), sc->radius ) )
      {
        return TRUE;
      }
      return FALSE;

    } else {
      SPoint p;
      p.lng = p.lat = 0.0;
      if ( FPeq( spoint_dist ( &p, &tc.center ), sc->radius) ){
        return true;
      }
      p.lng = sl->length;
      if ( FPeq( spoint_dist ( &p, &tc.center ), sc->radius) ){
        return true;
      }
      return false;

    }

  }

  int8 sline_sline_pos ( const SLine * l1, const SLine * l2  )
  {

    static SEuler se;
    static SLine  sl1 , sl2 ;
    static SPoint p[4];
    static bool   a1, a2, a3, a4, switched ;
    static float8 i, k, mi, mk;
    static const float8 step = ( PI-0.1 );
    static int    res ;

    switched = FALSE;

    if ( sline_eq ( l1, l2 ) ){
      return PGS_LINE_EQUAL;
    }
    sline_swap_beg_end( &sl1, l1 );
    if ( sline_eq ( &sl1, l2 ) ){
      return PGS_LINE_CONT_LINE;
    }
    
    // transform the larger line into equator ( begin at (0,0) )
    sl1.phi = sl1.theta = sl1.psi = 0.0;
    if ( FPge( l1->length, l2->length ) ){
      sphereline_to_euler_inv ( &se, l1 );
      sl1.length = l1->length;
      euler_sline_trans ( &sl2 , l2 , &se );
      switched = FALSE;
    } else
    if ( FPge( l2->length, l1->length ) ){
      sphereline_to_euler_inv ( &se, l2 ) ;
      sl1.length = l2->length;
      euler_sline_trans ( &sl2 , l1 , &se );
      switched = TRUE;
    }
    if ( FPzero( sl1.length ) ){ // both are points
      return PGS_LINE_AVOID;
    }

    res = 0;

    // check connected lines

    if ( FPgt( sl2.length, 0.0 ) ){
      sline_begin ( &p[0], &sl1 );
      sline_end   ( &p[1], &sl1 );
      sline_begin ( &p[2], &sl2 );
      sline_end   ( &p[3], &sl2 );

      if ( spoint_eq (&p[0],&p[2]) ) res = ( 1 << PGS_LINE_CONNECT );
      if ( spoint_eq (&p[0],&p[3]) ) res = ( 1 << PGS_LINE_CONNECT );
      if ( spoint_eq (&p[1],&p[2]) ) res = ( 1 << PGS_LINE_CONNECT );
      if ( spoint_eq (&p[1],&p[3]) ) res = ( 1 << PGS_LINE_CONNECT );
    }

    // split lines in segments less than 180 degrees and check for each of it

    mi = sl1.length / step ;
    mk = sl2.length / step ;
    if ( FPzero ( mk ) ){
      mk = 0.1; // force one loop for sl2
    }


    sl2.psi    += step ;
    for ( i=0.0; i<mi; i += 1.0 ){

      sl2.psi    -= step ;
      if ( ( i+1 ) >= mi ){
        sline_point_by_length( &p[0], &sl1, 0.0 );
        sline_point_by_length( &p[1], &sl1, sl1.length - (i*step) );
      } else
      if ( i == 0.0 ){
        sline_point_by_length( &p[0], &sl1, 0.0  );
        sline_point_by_length( &p[1], &sl1, step );
      }

      for ( k=0.0; k<mk; k += 1.0 ){
        sline_point_by_length( &p[2], &sl2, k * step );
        sline_point_by_length( &p[3], &sl2, 
          ( ( ( k+1.0 ) > mk )?( sl2.length ):( ( k+1.0 ) * step ) ) );
        a1  = ( FPge(p[2].lat,0.0) && FPle(p[3].lat,0.0) ); // sl2 crosses equator desc.
        a2  = ( FPle(p[2].lat,0.0) && FPge(p[3].lat,0.0) ); // sl1 crosses equator asc. 
        a3  = FPzero(p[2].lat) ;                            // begin of sl2 at equator  
        a4  = FPzero(p[3].lat) ;                            // end   of sl2 at equator  

        if ( ! ( a1 || a2  ) ) {
          res |= ( 1 << PGS_LINE_AVOID );

        } else {

          static Vector3D v[2][2] ;
          static bool lbeg, lend ; 

          // Now we take the vectors of line's begin and end
          spoint_vector3d ( &v[0][0], &p[0] );
          spoint_vector3d ( &v[0][1], &p[1] );
          spoint_vector3d ( &v[1][0], &p[2] );
          spoint_vector3d ( &v[1][1], &p[3] );

          if ( v[0][1].x <= 0.0 ){
            v[0][1].y = 1.0;
          }

          // check whether sl2's longitudes are in sl1 range ( begin and end )
          lbeg =   FPle( v[0][1].x, v[1][0].x ) && FPle ( v[1][0].x, 1.0 ) &&
                   FPle( 0.0, v[1][0].y ) && FPle ( v[1][0].y, v[0][1].y ) ; 
          lend =   FPle( v[0][1].x, v[1][1].x ) && FPle ( v[1][1].x, 1.0 ) &&
                   FPle( 0.0, v[1][1].y ) && FPle ( v[1][1].y, v[0][1].y ) ; 

          if ( a3 && a4 ) {       // sl2's begin and end at equator
            if ( res & ( 1 << PGS_LINE_CONNECT ) ) {
              return PGS_LINE_OVER;
            }
            if ( lbeg && lend ){  // sl2's begin and end at line
              if ( switched ){
                return PGS_LINE_OVER ;
              } else {
                res |= ( 1 << PGS_LINE_CONT_LINE );
              }
            } else
            if ( lbeg || lend ) { // begin xor end at line
              if ( switched ){
                return PGS_LINE_OVER ;
              } else
              if ( k==0.0 || (k+1.0)>=mk ){ // begin or end of sl2 line segment
                if ( FPlt(sl1.length,PID) && i==0.0 && ( FPlt ( v[1][0].y , 0.0 ) || FPlt ( v[1][1].y , 0.0 ) ) ){
                  return PGS_LINE_OVER ;
                } else
                if ( FPlt(sl1.length,PID) && (i+1.0)>=mi && ( FPlt ( v[1][0].x , v[0][1].x ) || FPlt ( v[1][1].x , v[0][1].x ) ) ){
                  return  PGS_LINE_OVER ;
                } else {
                  res |= ( 1 << PGS_LINE_CONT_LINE );
                }
              } else {
                res |= ( 1 << PGS_LINE_CONT_LINE );
              }
            } else { // begin and end not at line
              res |= ( 1 << PGS_LINE_AVOID );
            }
            
          } else { // begin and end ( both ) are not at equator

            SPoint sp;
            sphereline_to_euler_inv ( &se, &sl2 );
            sp.lng = ( ( a1 )?(PI):(0.0) ) - se.phi; // node
            sp.lat = 0;
            spoint_check ( &sp );
            if (  FPge(sp.lng,0.0) && FPle(sp.lng,p[1].lng) ){
              res |= ( 1 << PGS_LINE_CROSS );
            } else {
              res |= ( 1 << PGS_LINE_AVOID );
            }
          }  
        }
      }
    }

    if ( res & ( 1 << PGS_LINE_CONNECT ) ){
      return PGS_LINE_CONNECT;
    }
    if ( res & ( 1 << PGS_LINE_CONT_LINE ) ){
      return PGS_LINE_CONT_LINE;
    }
    if ( res == ( 1 << PGS_LINE_AVOID ) ){
      return PGS_LINE_AVOID;
    }
    if ( res == ( 1 << PGS_LINE_CROSS ) ){
      return PGS_LINE_CROSS;
    }

    return PGS_LINE_AVOID;

  }

  SEuler * sphereline_to_euler_inv ( SEuler * se, const SLine * sl )
  {
    sphereline_to_euler ( se, sl );
    spheretrans_inv ( se );
    return se;
  }

  SEuler * sphereline_to_euler ( SEuler * se, const SLine * sl )
  {
     seuler_set_zxz ( se );
     se->phi   = sl->phi;
     se->theta = sl->theta;
     se->psi   = sl->psi;
     return se;
  }


  SLine  * euler_sline_trans ( SLine * out , const SLine  * in , const SEuler * se )
  {
    static SEuler stmp[2];
    sphereline_to_euler ( &stmp[0], in );
    seuler_trans_zxz ( &stmp[1] , &stmp[0] , se );
    out->phi    = stmp[1].phi;
    out->theta  = stmp[1].theta;
    out->psi    = stmp[1].psi;
    out->length = in->length;
    return out;
  }

  bool spoint_at_sline( const SPoint * p, const SLine * sl )
  {
    static SLine l;
    sline_from_points ( &l, p, p );
    return ( sline_sline_pos ( sl, &l  ) == PGS_LINE_CONT_LINE ) ;
  }

  SPoint * sline_center( SPoint * c, const SLine * sl )
  {
    static SEuler se;
    static SPoint p;
    p.lng = sl->length/2.0;
    p.lat = 0.0;
    sphereline_to_euler ( &se, sl );
    return ( euler_spoint_trans ( c , &p , &se ) );
  }






  Datum  sphereline_in(PG_FUNCTION_ARGS)
  {
    SLine     * sl  = ( SLine * ) MALLOC ( sizeof ( SLine ) ) ;
    char      *  c  = PG_GETARG_CSTRING(0);
    unsigned char      etype[3] ;
    float8      eang[3], length ;
    SEuler        se , stmp, so ;
    int                       i ;

    void sphere_yyparse( void );

    init_buffer ( c );
    sphere_yyparse(); 
    if ( get_line ( &eang[0], &eang[1], &eang[2], etype, &length ) ){

      for ( i=0; i<3; i++ ){
        switch ( i ){
          case 0:  se.phi_a   = etype[i] ; break;
          case 1:  se.theta_a = etype[i] ; break;
          case 2:  se.psi_a   = etype[i] ; break;
        }
      }
      se.phi     = eang[0] ;
      se.theta   = eang[1] ;
      se.psi     = eang[2] ;

      stmp.phi   = stmp.theta   = stmp.psi   = 0.0;
      stmp.phi_a = stmp.theta_a = stmp.psi_a = EULER_AXIS_Z;      
      seuler_trans_zxz ( &so , &se , &stmp );

      sl->phi    = so.phi;
      sl->theta  = so.theta;
      sl->psi    = so.psi;

      if ( FPge(length, PID) ){
        length = PID ;
      }
      sl->length = length;

    } else {
      reset_buffer();
      FREE( sl );
      sl = NULL;
      elog ( ERROR , "sphereline_in: parse error" );
    }
    reset_buffer();

    PG_RETURN_POINTER( sl );

  }

  Datum  sphereline_equal(PG_FUNCTION_ARGS)
  {
    SLine  * l1 =  ( SLine * )  PG_GETARG_POINTER ( 0 ) ;
    SLine  * l2 =  ( SLine * )  PG_GETARG_POINTER ( 1 ) ;
    PG_RETURN_BOOL ( sline_eq ( l1, l2 ) );
  }

  Datum  sphereline_equal_neg(PG_FUNCTION_ARGS)
  {
    SLine  * l1 =  ( SLine * )  PG_GETARG_POINTER ( 0 ) ;
    SLine  * l2 =  ( SLine * )  PG_GETARG_POINTER ( 1 ) ;
    PG_RETURN_BOOL ( !sline_eq ( l1, l2 ) );
  }

  Datum  sphereline_from_point (PG_FUNCTION_ARGS)
  {
    SLine   * sl  = ( SLine   * ) MALLOC ( sizeof ( SLine ) ) ;
    SPoint  * p   = ( SPoint  * ) PG_GETARG_POINTER( 0 );
    sline_from_points ( sl, p, p );
    PG_RETURN_POINTER( sl );
  }

  Datum  sphereline_from_points (PG_FUNCTION_ARGS)
  {
    SLine   * sl  = ( SLine   * ) MALLOC ( sizeof ( SLine ) ) ;
    SPoint  * beg = ( SPoint  * ) PG_GETARG_POINTER( 0 );
    SPoint  * end = ( SPoint  * ) PG_GETARG_POINTER( 1 );

    if ( ! sline_from_points ( sl, beg, end ) ){
      FREE( sl );
      sl = NULL;
      elog ( ERROR , "sphereline_from_points: length of line must be not equal 180 degrees" );
    }

    PG_RETURN_POINTER( sl );
  }

  Datum  sphereline_from_trans (PG_FUNCTION_ARGS)
  {

    SLine   * sl   = ( SLine   * ) MALLOC ( sizeof ( SLine ) ) ;
    SEuler  * se   = ( SEuler  * ) PG_GETARG_POINTER( 0 );
    float8     l   = PG_GETARG_FLOAT8( 1 );
    if ( FPlt ( l,0.0 ) ){
      FREE ( sl );
      elog ( ERROR , "sphereline_from_trans: length of line must be >= 0" );
      PG_RETURN_NULL( );
    } else {
      static SEuler tmp;
      if ( FPgt ( l, PID ) ){
        l = PID;
      }
      strans_zxz(&tmp, se);
      sl->phi    = tmp.phi;
      sl->theta  = tmp.theta;
      sl->psi    = tmp.psi;
      sl->length = l;
    }

    PG_RETURN_POINTER( sl );
  }


  Datum  sphereline_meridian (PG_FUNCTION_ARGS)
  {
    SLine  * out =  ( SLine  * )  MALLOC ( sizeof ( SLine ) );
    float8 lng   = PG_GETARG_FLOAT8( 0 );
    sline_meridian ( out, lng );
    PG_RETURN_POINTER ( out );
  }

  Datum  sphereline_swap_beg_end (PG_FUNCTION_ARGS)
  {
    SLine  * in  =  ( SLine  * )  PG_GETARG_POINTER ( 0 ) ;
    SLine  * out =  ( SLine  * )  MALLOC ( sizeof ( SLine ) );
    sline_swap_beg_end ( out, in );
    PG_RETURN_POINTER ( out );
  }

  Datum  sphereline_begin(PG_FUNCTION_ARGS)
  {
    SLine  * sl =  ( SLine  * )  PG_GETARG_POINTER ( 0 ) ;
    SPoint * sp =  ( SPoint * )  MALLOC ( sizeof ( SPoint ) );
    sline_begin ( sp, sl );
    PG_RETURN_POINTER ( sp );
  }

  Datum  sphereline_end(PG_FUNCTION_ARGS)
  {
    SLine  * sl =  ( SLine * )  PG_GETARG_POINTER ( 0 ) ;
    SPoint * sp =  ( SPoint * )  MALLOC ( sizeof ( SPoint ) );
    sline_end ( sp, sl );
    PG_RETURN_POINTER ( sp );
  }

  Datum  sphereline_length(PG_FUNCTION_ARGS)
  {
    SLine    * sl =  ( SLine * )  PG_GETARG_POINTER ( 0 ) ;
    PG_RETURN_FLOAT8 ( sl->length );
  }


  Datum spherecircle_cont_line (PG_FUNCTION_ARGS)
  {
    SCIRCLE  * c  =  ( SCIRCLE * ) PG_GETARG_POINTER ( 0 ) ;
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 1 ) ;
    int8       b  =  sphereline_circle_pos ( l , c );
          
    PG_RETURN_BOOL ( b == PGS_CIRCLE_CONT_LINE );
  }

  Datum spherecircle_cont_line_neg (PG_FUNCTION_ARGS)
  {
    SCIRCLE  * c  =  ( SCIRCLE * ) PG_GETARG_POINTER ( 0 ) ;
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 1 ) ;
    int8       b  =  sphereline_circle_pos ( l , c );

    PG_RETURN_BOOL ( b != PGS_CIRCLE_CONT_LINE );
  }

  Datum spherecircle_cont_line_com (PG_FUNCTION_ARGS)
  {
    SCIRCLE  * c  =  ( SCIRCLE * ) PG_GETARG_POINTER ( 1 ) ;
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 0 ) ;
    int8       b  =  sphereline_circle_pos ( l , c );
          
    PG_RETURN_BOOL ( b == PGS_CIRCLE_CONT_LINE );
  }

  Datum spherecircle_cont_line_com_neg (PG_FUNCTION_ARGS)
  {
    SCIRCLE  * c  =  ( SCIRCLE * ) PG_GETARG_POINTER ( 1 ) ;
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 0 ) ;
    int8       b  =  sphereline_circle_pos ( l , c );

    PG_RETURN_BOOL ( b != PGS_CIRCLE_CONT_LINE );
  }

  Datum sphereline_overlap_circle (PG_FUNCTION_ARGS)
  {
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 0 ) ;
    SCIRCLE  * c  =  ( SCIRCLE * ) PG_GETARG_POINTER ( 1 ) ;
    int8       b  =  sphereline_circle_pos ( l , c );
    PG_RETURN_BOOL ( b > PGS_CIRCLE_LINE_AVOID );
  }

  Datum sphereline_overlap_circle_neg (PG_FUNCTION_ARGS)
  {
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 0 ) ;
    SCIRCLE  * c  =  ( SCIRCLE * ) PG_GETARG_POINTER ( 1 ) ;
    int8       b  =  sphereline_circle_pos ( l , c );
    PG_RETURN_BOOL ( b == PGS_CIRCLE_LINE_AVOID );
  }

  Datum sphereline_overlap_circle_com (PG_FUNCTION_ARGS)
  {
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 1 ) ;
    SCIRCLE  * c  =  ( SCIRCLE * ) PG_GETARG_POINTER ( 0 ) ;
    int8       b  =  sphereline_circle_pos ( l , c );
    PG_RETURN_BOOL ( b > PGS_CIRCLE_LINE_AVOID );
  }

  Datum sphereline_overlap_circle_com_neg (PG_FUNCTION_ARGS)
  {
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 1 ) ;
    SCIRCLE  * c  =  ( SCIRCLE * ) PG_GETARG_POINTER ( 0 ) ;
    int8       b  =  sphereline_circle_pos ( l , c );
    PG_RETURN_BOOL ( b == PGS_CIRCLE_LINE_AVOID );
  }

  Datum sphereline_crosses (PG_FUNCTION_ARGS)
  {
    SLine    * l1  =  ( SLine   * ) PG_GETARG_POINTER ( 0 ) ;
    SLine    * l2  =  ( SLine   * ) PG_GETARG_POINTER ( 1 ) ;
    int8        r  = sline_sline_pos ( l1, l2  );
    PG_RETURN_BOOL ( r == PGS_LINE_CROSS );
  }

  Datum sphereline_crosses_neg (PG_FUNCTION_ARGS)
  {
    SLine    * l1  =  ( SLine   * ) PG_GETARG_POINTER ( 0 ) ;
    SLine    * l2  =  ( SLine   * ) PG_GETARG_POINTER ( 1 ) ;
    int8        r  = sline_sline_pos ( l1, l2  );
    PG_RETURN_BOOL ( r != PGS_LINE_CROSS );
  }

  Datum sphereline_overlap (PG_FUNCTION_ARGS)
  {
    SLine    * l1  =  ( SLine   * ) PG_GETARG_POINTER ( 0 ) ;
    SLine    * l2  =  ( SLine   * ) PG_GETARG_POINTER ( 1 ) ;
    PG_RETURN_BOOL ( sline_sline_pos ( l1, l2  ) > PGS_LINE_AVOID );
  }

  Datum sphereline_overlap_neg (PG_FUNCTION_ARGS)
  {
    SLine    * l1  =  ( SLine   * ) PG_GETARG_POINTER ( 0 ) ;
    SLine    * l2  =  ( SLine   * ) PG_GETARG_POINTER ( 1 ) ;
    PG_RETURN_BOOL ( sline_sline_pos ( l1, l2  ) == PGS_LINE_AVOID );
  }

  Datum sphereline_cont_point (PG_FUNCTION_ARGS)
  {
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 0 ) ;
    SPoint   * p  =  ( SPoint  * ) PG_GETARG_POINTER ( 1 ) ;
    PG_RETURN_BOOL ( spoint_at_sline( p, l ) );
  }

  Datum sphereline_cont_point_neg (PG_FUNCTION_ARGS)
  {
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 0 ) ;
    SPoint   * p  =  ( SPoint  * ) PG_GETARG_POINTER ( 1 ) ;
    PG_RETURN_BOOL ( ! spoint_at_sline( p, l ) );
  }

  Datum sphereline_cont_point_com (PG_FUNCTION_ARGS)
  {
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 1 ) ;
    SPoint   * p  =  ( SPoint  * ) PG_GETARG_POINTER ( 0 ) ;
    PG_RETURN_BOOL ( spoint_at_sline( p, l ) );
  }

  Datum sphereline_cont_point_com_neg (PG_FUNCTION_ARGS)
  {
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 1 ) ;
    SPoint   * p  =  ( SPoint  * ) PG_GETARG_POINTER ( 0 ) ;
    PG_RETURN_BOOL ( ! spoint_at_sline( p, l ) );
  }

  Datum  spheretrans_line(PG_FUNCTION_ARGS) {
  
    SLine   * sl =  ( SLine  * ) PG_GETARG_POINTER ( 0 ) ;
    SEuler  * se =  ( SEuler * ) PG_GETARG_POINTER ( 1 ) ;
    SLine   *ret =  ( SLine  * ) MALLOC ( sizeof ( SLine ) );
    PG_RETURN_POINTER ( euler_sline_trans ( ret, sl, se ) );
  }

  Datum  spheretrans_line_inverse(PG_FUNCTION_ARGS)
  {

   Datum sl     = PG_GETARG_DATUM ( 0 );
   SEuler *  se = ( SEuler * ) PG_GETARG_POINTER( 1 );
   SEuler tmp   ;
   Datum ret;
   
   spheretrans_inverse ( &tmp , se );
   ret =  DirectFunctionCall2(
                       spheretrans_line ,
                       sl, PointerGetDatum(&tmp) );
   PG_RETURN_DATUM( ret );

  }

  Datum spheretrans_from_line (PG_FUNCTION_ARGS)
  {
    SLine    * l  =  ( SLine   * ) PG_GETARG_POINTER ( 0 ) ;
    SEuler   * e  =  ( SEuler  * ) MALLOC ( sizeof ( SEuler ) ) ;
    sphereline_to_euler ( e, l );
    PG_RETURN_POINTER ( e );
  }

