# $Id: payment.pm,v 1.16 2004/02/07 06:46:33 mig Exp $
######################################
# Comas - Conference Management System
######################################
# Copyright 2003 CONSOL
# Congreso Nacional de Software Libre (http://www.consol.org.mx/)
#   Gunnar Wolf <gwolf@gwolf.cx>
#   Manuel Rabade <mig@mig-29.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################

######################################
# Module: Comas::HTML::committee::payment
# Front End for the payment tasks
######################################
# Depends on:

# Comas::Person - Handles the interaction with a person for Comas
# Comas::HTML::common - Common functions for the HTML FrontEnd
# Comas::Proposal - Handles the interaction with a person for Comas
# Comas::Admin - Manage administrative tasks for a Comas database 
# HTML::Template - Perl module to use HTML Templates
# Data::FormValidator - Validates user input based on input profile

# This module is not meant to be called on its own, it should be called from
# Comas::HTML.

package Comas::HTML::committee::payment;

use strict;
use warnings;
use Carp;

use Comas::Admin;
use Comas::Conf qw(:read);
use Comas::Person;
use Comas::Proposal;
use Comas::HTML::common qw(:all);
use Comas::Payment;

use HTML::Template;
use Data::FormValidator;
use PDF::API2;

use Exporter;

our @ISA = qw(Exporter);
our @EXPORT_OK = qw(payment_reg_select_person payment_reg_search_person
                    payment_reg_select payment_reg_confirm payment_reg_register

                    payment_mod_select_person_delete
                    payment_mod_search_person_delete payment_mod_select_delete
                    payment_mod_confirm_delete payment_mod_delete

                    payment_print_sticker payment_print_diploma);

our %EXPORT_TAGS = (register => [qw(payment_reg_select_person
                                    payment_reg_search_person payment_reg_select
                                    payment_reg_confirm payment_reg_register)],

                    modify => [qw(payment_mod_select_person_delete
                                  payment_mod_search_person_delete
                                  payment_mod_select_delete
                                  payment_mod_confirm_delete
                                  payment_mod_delete )],

                    util => [qw( payment_print_sticker
                                 payment_print_diploma )],

                    all => [@EXPORT_OK] );

=head1 NAME
    
Comas::HTML::committee::payment - Front End for the monetary admin tasks

=head1 SYNOPSIS

This module is not meant to be used by itself, but as an auxiliary to
L<Comas::HTML|Comas::HTML>. Please check L<Comas::HTML|Comas::HTML>'s
documentation, in the B<Front End Tasks> section, for further details on the
usage for this functions.

=head1 SEE ALSO

L<Comas::HTML|Comas::HTML> module documentation

=head1 AUTHOR

Gunnar Wolf, gwolf@gwolf.cx

Manuel Rabade, mig@mig-29.net

Comas has been developed for CONSOL, Congreso Nacional de Software Libre,
http://www.consol.org.mx/

=head1 COPYRIGHT

Copyright 2003 Gunnar Wolf and Manuel Rabade

This library is free software, you can redistribute it and/or modify it
under the terms of the GPL version 2 or later.

=cut

sub payment_reg_select_person {
    my $h = shift;

    my ($template, $person) = $h->_select_person;
    $template->param(-title => 'Registrar un pago');
    if ($person) {
        return $h->payment_reg_select($person);
    } else {
        return $template->output;
    }
}

sub payment_reg_search_person {
    my ($template, $input_profile, $validator, $val_results, @person_ids,
        @persons);

    my $h = shift;
    $template = $h->_search_person;
    $template->param(-title => 'Registrar un pago');
    return $template->output;
}

sub payment_reg_select {
    my (@proposal_ids, @proposals, $template, @proposals_sorted, $person,
        %person_data, %general_entry, %old_payments);
    my $h = shift;
    my $person_id = shift;

    unless($h->_ck_committee_task) {
        return $h->committee_menu('NOT_AUTHORIZED');
    }

    $template = HTML::Template->new(filename=>
                                    'committee/payment/select_payment.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);
    
    unless (defined $person_id) { $person_id = $h->{-path}->[3]; }
    
    $person = Comas::Person->new(-db => $h->{-db}, 
                                 -id => $person_id);
    
    %person_data = $person->get_data;
    $template->param(-id => $person->{-id},
                     -firstname => $person_data{-firstname},
                     -famname => $person_data{-famname},
                     -email => $person_data{-email},
                     -person_type =>
                     _id2descr($h->{-db},'person_type',
                               $person_data{-person_type_id}));
    
    foreach my $payment (Comas::Payment->get_payments(-db => $h->{-db},
                                                      -person =>
                                                      $person->{-id})) {
        my (undef, $prop_id) = Comas::Payment->get_payment_data(-db =>
                                                                $h->{-db},
                                                                -payment_id =>
                                                                $payment);
        if (defined($prop_id)) {
            $old_payments{$prop_id} = 1;
        } else {
            $old_payments{0} = 1;
        }            
    }

    @proposal_ids = Comas::Proposal->search(-db => $h->{-db},
                                            -prop_status_id =>
                                            get_conf($h->{-db},
                                                     'proposals_accepted_id'));
    foreach my $p_id (@proposal_ids) {
        my ($proposal_temp, %row, %prop_data);
        $proposal_temp = Comas::Proposal->new(-db => $h->{-db},
                                              -id => $p_id);
        %prop_data = $proposal_temp->get_data;
        $row{amount} = Comas::Payment->calculate_amount(-db => $h->{-db},
                                                        -person => 
                                                        $person->{-id},
                                                        -proposals =>
                                                        [$prop_data{-id}]);
        if ($row{amount} == 0) { next; }
        
        $row{id} = $prop_data{-id};
        $row{title} = $prop_data{-title};
        $row{prop_type_id} = $prop_data{-prop_type_id};
        $row{prop_type} = _id2descr($h->{-db}, 'prop_type',
                                    $prop_data{-prop_type_id});
        ($row{people},
         $row{max_people}) = Comas::Payment->get_capacity(-db => $h->{-db},
                                                          -proposal =>
                                                          $prop_data{-id});
        if ($row{people} >= $row{max_people} and $row{max_people} != 0) {
            $row{full} = 1;
        }
        if ($old_payments{$row{id}}) {
            $row{payed} = 1;
        }
        push (@proposals, \%row);
    }
    
    @proposals_sorted = sort { $a->{title} cmp $b->{title} } @proposals;
    
    $general_entry{id} = 0;
    $general_entry{title} = _id2descr($h->{-db},'prop_type',0);
    $general_entry{amount} = Comas::Payment->calculate_amount(-db => $h->{-db},
                                                              -person => 
                                                              $person->{-id});

    ($general_entry{people}, $general_entry{max_people}) = (0,0);

    if ($old_payments{0}) {
        $general_entry{payed} = 1;
    }
    unless ($general_entry{amount} == 0) { push(@proposals_sorted,
                                           \%general_entry); }
    
    $template->param(PROPOSALS => \@proposals_sorted);
    return $template->output;
}

sub payment_reg_confirm {
    my ($person, $template, @proposals, @proposals_sorted, %person_data);
    my $h = shift;

    unless($h->_ck_committee_task) {
        return $h->committee_menu('NOT_AUTHORIZED');
    }

    $template = HTML::Template->new(filename=>
                                    'committee/payment/confirm_payment.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);

    $template->param(-payment_type_list => 
                     _catalog2html_opts($h->{-db},"payment_type",
                                        $h->{-params}->{-payment_type_id},
                                        undef,1));

    $person = Comas::Person->new(-db => $h->{-db}, 
                                 -id => $h->{-path}->[3]);
    %person_data = $person->get_data;
    $template->param(-id => $person->{-id},
                     -firstname => $person_data{-firstname},
                     -famname => $person_data{-famname},
                     -email => $person_data{-email},
                     -person_type =>
                     _id2descr($h->{-db},'person_type',
                               $person_data{-person_type_id}));

    $template->param(general_amount =>
                     Comas::Payment->calculate_amount(-db => $h->{-db},
                                                      -person => 
                                                      $person->{-id}),
                     general_title => _id2descr($h->{-db},'prop_type',0));
    
    foreach my $param (keys %{$h->{-params}}) {
        if ($param =~ /^pay_prop_(.*)/) {
            my ($proposal_temp, %row, %prop_data);
            if ($1 == 0) {
                $row{id} = 0;
                $row{title} = _id2descr($h->{-db},'prop_type',0);
                $row{amount} = Comas::Payment->
                    calculate_amount(-db => $h->{-db},
                                     -person =>$person->{-id});     
                ($row{people}, $row{max_people}) = (0,0);
            } else {
                $proposal_temp = Comas::Proposal->new(-db => $h->{-db},
                                                      -id => $1);
                %prop_data = $proposal_temp->get_data;
                $row{amount} = Comas::Payment->calculate_amount(-db => $h->{-db},
                                                                -person => 
                                                                $person->{-id},
                                                                -proposals =>
                                                                [$prop_data{-id}]);
                ($row{people},
                 $row{max_people}) = Comas::Payment->get_capacity(-db => 
                                                                  $h->{-db},
                                                                  -proposal =>
                                                                  $prop_data{-id});
                
                $row{id} = $prop_data{-id};
                $row{title} = $prop_data{-title};
                $row{prop_type_id} = $prop_data{-prop_type_id};
                $row{prop_type} = _id2descr($h->{-db}, 'prop_type',
                                            $prop_data{-prop_type_id});
            }
            push (@proposals, \%row);
        }
    }
    @proposals_sorted = sort { $a->{title} cmp $b->{title} } @proposals;
    $template->param(PROPOSALS => \@proposals_sorted);
    return $template->output;    
}

sub payment_reg_register {
    my ($person, $template, %person_data, @proposals, @proposals_sorted,
    $total_amount);
    my $h = shift;

    unless($h->_ck_committee_task) {
        return $h->committee_menu('NOT_AUTHORIZED');
    }

    $person = Comas::Person->new(-db => $h->{-db}, 
                                 -id => $h->{-path}->[3]);

    $template = HTML::Template->new(filename=>
                                    'committee/payment/register_payment.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);
    %person_data = $person->get_data;
    
    $template->param(-id => $person->{-id},
                     -firstname => $person_data{-firstname},
                     -famname => $person_data{-famname},
                     -email => $person_data{-email},
                     -person_type =>
                     _id2descr($h->{-db},'person_type',
                               $person_data{-person_type_id}));
    
    foreach my $param (keys %{$h->{-params}}) {
        my ($proposal_temp, %row, %prop_data);
        if ($param =~ /^amount_prop_(.*)/) {
            if ($1 == 0) {
                if (Comas::Payment->register(-db => $h->{-db_adm},
                                             -admin =>
                                             $h->{-session}->{-admin}->{-login},
                                             -admin_pwd =>
                                             $h->{-session}->{-admin}->{-passwd},
                                             -person => $person->{-id},
                                             -amount => $h->{-params}->{$param},
                                             -payment_type =>
                                             $h->{-params}->{-payment_type_id})) {
                    $row{id} = 0;
                    $row{title} = _id2descr($h->{-db},'prop_type',0);
                    $row{amount} = $h->{-params}->{$param};
                    $total_amount += $row{amount};
                    push (@proposals, \%row);
                }
              } else {
                  warn $h->{-params}->{$param};
                  if (Comas::Payment->register(-db => $h->{-db_adm},
                                               -admin =>
                                               $h->{-session}->{-admin}->{-login},
                                               -admin_pwd =>
                                               $h->{-session}->{-admin}->{-passwd},
                                               -person => $person->{-id},
                                               -amount => $h->{-params}->{$param},
                                               -payment_type =>
                                               $h->{-params}->{-payment_type_id},
                                               -proposal => $1 )) {
                      $proposal_temp = Comas::Proposal->new(-db => $h->{-db},
                                                            -id => $1);
                      %prop_data = $proposal_temp->get_data;
                      $row{id} = $prop_data{-id};
                      $row{title} = $prop_data{-title};
                      $row{prop_type_id} = $prop_data{-prop_type_id};
                      $row{prop_type} = _id2descr($h->{-db}, 'prop_type',
                                                  $prop_data{-prop_type_id});
                      $row{amount} = $h->{-params}->{$param};
                      $total_amount += $row{amount};
                      push (@proposals, \%row);
                  }
              }
        }
    }
    @proposals_sorted = sort { $a->{title} cmp $b->{title} } @proposals;
    $template->param(PROPOSALS => \@proposals_sorted,
                     -total_amount => $total_amount,
                     -payment_type => 
                     _id2descr($h->{-db},'payment_type',
                               $h->{-params}->{-payment_type_id}));
    return $template->output;
}

###############################################################################

sub payment_mod_select_person_delete {
    my $h = shift;

    my ($template, $person) = $h->_select_person;
    $template->param(-title => 'Consulta y eliminacion de pagos');
    if ($person) {
        return $h->payment_mod_select_delete($person);
    } else {
        return $template->output;
    }
}

sub payment_mod_search_person_delete {
    my ($template, $input_profile, $validator, $val_results, @person_ids,
        @persons);

    my $h = shift;
    $template = $h->_search_person;
    $template->param(-title => 'Consulta y eliminacion de pagos');
    return $template->output;
}

sub payment_mod_select_delete {
    my $h = shift;
    my $person_id = shift;
    my $message = shift;

    my ($template, $person, %person_data, @payments, $total);

    unless($h->_ck_committee_task) {
        return $h->committee_menu('NOT_AUTHORIZED');
    }

    $template = HTML::Template->new(filename=>
                                    'committee/payment/delete_payment.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);

    if (defined $message) { $template->param($message => 1); }    

    unless (defined $person_id) { $person_id = $h->{-path}->[3]; }
    
    $person = Comas::Person->new(-db => $h->{-db}, 
                                 -id => $person_id);
    
    %person_data = $person->get_data;
    $template->param(-id => $person->{-id},
                     -firstname => $person_data{-firstname},
                     -famname => $person_data{-famname},
                     -email => $person_data{-email},
                     -person_type =>
                     _id2descr($h->{-db},'person_type',
                               $person_data{-person_type_id}));
    
    foreach my $payment (Comas::Payment->get_payments(-db => $h->{-db},
                                                      -person =>
                                                      $person->{-id})) {
        my ($proposal_temp, %row, %prop_data, $prop_id);

        (undef, $prop_id,
         $row{amount}) = Comas::Payment->get_payment_data(-db =>
                                                          $h->{-db},
                                                          -payment_id =>
                                                          $payment);

        $total += $row{amount};
        $row{id} = $payment;

        if ($prop_id == 0) {
            $row{title} = _id2descr($h->{-db},'prop_type',0);
            ($row{people}, $row{max_people}) = (0,0);
        } else {
            $proposal_temp = Comas::Proposal->new(-db => $h->{-db},
                                                  -id => $prop_id);
            %prop_data = $proposal_temp->get_data;
            
            $row{title} = $prop_data{-title};
            $row{prop_type_id} = $prop_data{-prop_type_id};
            $row{prop_type} = _id2descr($h->{-db}, 'prop_type',
                                        $prop_data{-prop_type_id});
            ($row{people},
             $row{max_people}) = Comas::Payment->get_capacity(-db => $h->{-db},
                                                              -proposal =>
                                                              $prop_data{-id});
        }
        push (@payments, \%row);
    }
    $template->param(PAYMENTS => \@payments,
                     -total => $total);
    return $template->output;
}

sub payment_mod_confirm_delete {
    my $h = shift;

    unless($h->_ck_committee_task) {
        return $h->committee_menu('NOT_AUTHORIZED');
    }

    my ($template, $person, %person_data, @payments, $total);

    $template = HTML::Template->new(filename=>
                                    'committee/payment/confirm_delete_payment.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);

    $person = Comas::Person->new(-db => $h->{-db}, 
                                 -id => $h->{-path}->[3]);

    %person_data = $person->get_data;

    $template->param(-id => $person->{-id},
                     -firstname => $person_data{-firstname},
                     -famname => $person_data{-famname},
                     -email => $person_data{-email},
                     -person_type =>
                     _id2descr($h->{-db},'person_type',
                               $person_data{-person_type_id}));

    foreach my $param (keys %{$h->{-params}}) {
        if ($param =~ /^del_pay_(.*)/) {
            my ($proposal_temp, %row, %prop_data, $prop_id);

            (undef, $prop_id,
             $row{amount}) = Comas::Payment->get_payment_data(-db =>
                                                              $h->{-db},
                                                              -payment_id =>
                                                              $1);
            
            $total += $row{amount};
            $row{id} = $1;

            if ($prop_id == 0) {
                $row{title} = _id2descr($h->{-db},'prop_type',0);
                ($row{people}, $row{max_people}) = (0,0);                
            } else {
                $proposal_temp = Comas::Proposal->new(-db => $h->{-db},
                                                      -id => $prop_id);
                %prop_data = $proposal_temp->get_data;
                
                $row{title} = $prop_data{-title};
                $row{prop_type_id} = $prop_data{-prop_type_id};
                $row{prop_type} = _id2descr($h->{-db}, 'prop_type',
                                            $prop_data{-prop_type_id});
                ($row{people},
                 $row{max_people}) = Comas::Payment->get_capacity(-db => $h->{-db},
                                                                  -proposal =>
                                                                  $prop_data{-id});
            }
            push (@payments, \%row);
        }
    }
    
    $template->param(PAYMENTS => \@payments,
                     -total => $total);
    return $template->output;
}

sub payment_mod_delete {
    my $h = shift;

    my ($template, $person, %person_data, @payments, $total);

    unless($h->_ck_committee_task) {
        return $h->committee_menu('NOT_AUTHORIZED');
    }

    foreach my $param (keys %{$h->{-params}}) {
        if ($param =~ /^del_pay_(.*)/) {
            Comas::Payment->delete(-db=> $h->{-db_adm}, -id => $1);
          }
    }

    return $h->payment_mod_select_delete($person,'DELETED');    
}

###############################################################################

sub payment_print_sticker {
    my $h = shift;
    my $person;
    unless($h->_ck_committee_task) {
        return $h->committee_menu('NOT_AUTHORIZED');
    }

    $person = Comas::Person->new(-db => $h->{-db}, -id => $h->{-path}->[3]);    
    _print_sticker($person, $h->{-config}->{-tmpl_path} . 'other/gafete.txt');
    return '<script type="text/javascript" language="JavaScript">history.go(-1)</script>';

}

sub _print_sticker {
    my ($person, $sticker_tmpl) = (shift, shift, shift);
    my (%sticker_subst, $sticker, %person_data);

    my %person_type_sticker = ( 1 => 'Asistente',
                                2 => 'Asistente',
                                3 => 'Organizador',
                                4 => 'Prensa',
                                5 => 'Patrocinador',
                                6 => 'Voluntario',
                                7 => 'Asistente',
                                8 => 'Ponente',
                                9 => 'Invitado' );
    
    unless (-r $sticker_tmpl and open(TMPL, '<', $sticker_tmpl)) {
        warn "Could not open sticker template ($sticker_tmpl): $!\n";
	warn 'Sticker will not be print';
	return undef;
    }
    
    while (<TMPL>) {
        $sticker .= $_;
    }
    
    close(TMPL);
    
    %person_data = $person->get_data;

    foreach my $key (keys %person_data) {
        if (defined $person_data{$key}) {
            $person_data{$key} =~
                y//YuAAAAAAACEEEEIIIIDNOOOOOOUUUUYsaaaaaaaceeeeiiiionoooooouuuuyy/;
        }
    }
    
    unless ($person_data{-nickname}) {
        ($sticker_subst{NICK}) = ($person_data{-firstname} =~ /^(\S+)/);
    } else {
        $sticker_subst{NICK} = $person_data{-nickname};
    }
    
    $sticker_subst{FIRSTNAME} = $person_data{-firstname};
    $sticker_subst{FAMNAME} = $person_data{-famname};
    $sticker_subst{EMAIL} = $person_data{-email};
    $sticker_subst{PERSON_TYPE} =
        $person_type_sticker{$person_data{-person_type_id}};
    
    for my $key (keys %sticker_subst) {
        $sticker =~ s/__${key}__/$sticker_subst{$key}/gs;
    }
    
    open(SERIAL_PORT, '>', '/dev/cuaa0');
    print SERIAL_PORT $sticker;
    close(SERIAL_PORT);
}

sub payment_print_diploma {
    my $h = shift;
    my $person;

    unless($h->_ck_committee_task) {
        return $h->committee_menu('NOT_AUTHORIZED');
    }

    $person = Comas::Person->new(-db => $h->{-db}, -id => $h->{-path}->[3]);    
    _print_diploma($person);
    return '<script type="text/javascript" language="JavaScript">history.go(-1)</script>';
}

sub _print_diploma {
    my $person = shift;

    my $printer = 'LaserJet6P';
    my $ancho = 612;
    my $alto = 792;    

    my %person_data = $person->get_data;
    my $person_name = $person_data{-firstname} . ' ' . $person_data{-famname};
    my $pdf=PDF::API2->new;
    my $font = $pdf->corefont('Helvetica',1);
    $font->encode('latin1');
    my $page = $pdf->page;
    $page->mediabox('letter');
    my $name = $page->text;
    $name->font($font, 30);

    my $iniciox = ($ancho/2)+75;

    my $inicioy = ($alto + 80 - $name->advancewidth($person_name))/2;

    $name->transform_rel(-translate=>[$iniciox, $inicioy],
			 -rotate=>90);

    $name->text($person_name);

    my $output = $pdf->stringify;
    open(OUT, "| /usr/local/bin/pdf2ps - - | /usr/local/bin/lpr -P$printer");
    print OUT $output;
    close(OUT);
    $pdf->end;
}

1;

# $Log: payment.pm,v $
# Revision 1.16  2004/02/07 06:46:33  mig
# - Al pedo el centrado del nombre en las constancias :)
#
# Revision 1.15  2004/02/07 01:12:53  mig
# - Corrigo mega bug cuando se borraban pagos no se tomaba en cuenta cuando
#   este era la entrada general.
#
# Revision 1.14  2004/02/07 00:12:46  mig
# - Lista la consulta y modificacion de pagos
#
# Revision 1.13  2004/02/06 02:12:47  mig
# - Una ruta completa es mejor :)
#
# Revision 1.12  2004/02/06 02:02:51  mig
# - Lista la impresion de constancias :)
#
# Revision 1.11  2004/02/05 21:43:01  mig
# - Despues de imprimir te regresa a la pagina anterior
#
# Revision 1.10  2004/02/05 21:17:49  mig
# - A ver si esta es la buena del nick :-P
#
# Revision 1.9  2004/02/05 21:07:02  mig
# - Ya quedo el pedo de cuando no existe nick :)
#
# Revision 1.8  2004/02/05 20:51:47  mig
# - Error cuando no hay nick
#
# Revision 1.7  2004/02/05 20:16:16  mig
# - Al parecer listo los gafetes :)
#
# Revision 1.6  2004/02/03 03:22:00  mig
# - Mayor control en la creacion de pagos.
#
# Revision 1.5  2004/01/30 07:24:49  mig
# - Correcciones de identado
#
# Revision 1.4  2004/01/24 01:43:32  mig
# - Lista la busqueda de personas :)
#
# Revision 1.3  2004/01/23 07:48:39  mig
# - Ya hay registro de pagos !
#
