# $Id: committee.pm,v 1.5 2003/12/20 04:14:51 mig Exp $
######################################
# Comas - Conference Management System
######################################
# Copyright 2003 CONSOL
# Congreso Nacional de Software Libre (http://www.consol.org.mx/)
#   Gunnar Wolf <gwolf@gwolf.cx>
#   Manuel Rabade <mig@mig-29.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################

######################################
# Module: Comas::HTML::committee
# Manage committee tasks.
######################################
# Depends on:
#
# Comas::Admin - Manage administrative tasks for a Comas database
# Comas::HTML::common - Common functions for the HTML FrontEnd
# HTML::Template - Perl module to use HTML Templates
# Data::FormValidator - Validates user input based on input profile

# This module is not meant to be called on its own, it should be called from
# Comas::HTML.

package Comas::HTML::committee;

use strict;
use warnings;
use Carp;

use Comas::Admin;
use Comas::HTML::common qw(:all);

use HTML::Template;
use Data::FormValidator;

use Exporter;

our @ISA = qw(Exporter);
our @EXPORT_OK = qw(committee_menu committee_login committee_logout
                    committee_passwd _ck_committee_task);
our %EXPORT_TAGS = (all => [@EXPORT_OK]);

=head1 NAME
    
Comas::HTML::committee - Front End to manage committee task's

=head1 SYNOPSIS

This module is not meant to be used by itself, but as an auxiliary to
L<Comas::HTML|Comas::HTML>. Please check L<Comas::HTML|Comas::HTML>'s
documentation, in the B<Front End Tasks> section, for further details on the
usage for this functions.

=head1 SEE ALSO

L<Comas::HTML|Comas::HTML> module documentation

=head1 AUTHOR

Gunnar Wolf, gwolf@gwolf.cx

Manuel Rabade, mig@mig-29.net

Comas has been developed for CONSOL, Congreso Nacional de Software Libre,
http://www.consol.org.mx/

=head1 COPYRIGHT

Copyright 2003 Gunnar Wolf and Manuel Rabade

This library is free software, you can redistribute it and/or modify it
under the terms of the GPL version 2 or later.

=cut

sub committee_menu {
    my ($template, $admin, %tasks);
    my $h = shift;
    my $message = shift;
    
    $template = HTML::Template->new(filename => 'committee/menu.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);
    $admin = Comas::Admin->new(-db => $h->{-db_adm},
                               %{$h->{-session}->{-admin}});
    
    $template->param(-name => $admin->get_name);
    
    if (defined $message) {
        $template->param($message => 1);
    }

    %tasks = $admin->get_tasks;

    foreach my $task (keys %tasks) {
        if($admin->ck_admin_task(-task=>$task, -login=>$admin->{-login})) {
            $template->param($task => 1);
        }
    }
    return $template->output;
}

sub committee_login {
    my ($input_profile, $template, $validator, $val_results, $admin);
    my $h = shift;
    
    $input_profile = {
        required => [ qw( -login -passwd ) ],
        filters => [ qw( trim ) ] };
    
    $template = HTML::Template->new(filename => 'committee/login.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);
    
    if (defined($h->{-params}->{validate})) {
        # If the form have been posted we check the values
        $validator = Data::FormValidator->new;
        $val_results = $validator->check ( $h->{-params}, $input_profile);
        
        if ($val_results->has_missing || $val_results->has_invalid) {
            # Wops, something is worng with the parameters
            $template->param(_validator2html($val_results));
            $template->param($h->{-params});
            return $template->output;
        } else {
            # We create a new person
            $admin = Comas::Admin->new(-db=>$h->{-db_adm},
                                       %{$val_results->valid});
            if (defined $admin) {
                $h->{-session}->{-admin} = $val_results->valid;
                return $h->committee_menu;
            } else {
                # Or we return the login form.
                $template->param(_validator2html($val_results));
                $template->param($h->{-params});
                $template->param(UNKNOWN_USER => '1');
                return $template->output;                                
            }
        }
    } else {
        # The form hasn't been posted.
        return $template->output;
    }
}

sub committee_logout {
    my $h = shift;
    delete $h->{-session}->{-admin};
    return $h->committee_login;
}

sub committee_passwd {
    my ($input_profile, $template, $validator, $val_results, $admin);
    my $h = shift;
    $input_profile = {
        required => [ qw( -current_passwd -new_passwd  -new_passwd2) ],
        filters => [ qw( trim ) ] };
    
    $template = HTML::Template->new(filename => 'committee/passwd.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);
    
    # This is all the validation if the form have been posted.
    if (defined($h->{-params}->{validate})) {
        $validator = Data::FormValidator->new;
        $val_results = $validator->check ( $h->{-params}, $input_profile);
        $template->param($h->{-params});        
        if ($val_results->has_missing || $val_results->has_invalid) {
            # If something is invalid
            $template->param(_validator2html($val_results));
            return $template->output;
        } elsif ($h->{-params}->{-new_passwd} ne $h->{-params}->{-new_passwd2}) {
            # If the passwords are diferent
            $template->param(DIFERENT_PASSWD => 1);
            return $template->output;
        } else {
            # Everything is fine ;)
            if($h->{-session}->{-admin}->{-passwd} eq
               $h->{-params}->{-current_passwd}) {
                $admin = Comas::Admin->new(-db => $h->{-db_adm},
                                           %{$h->{-session}->{-admin}});
                $admin->set_passwd($h->{-params}->{-new_passwd});
                $h->{-session}->{-admin}->{-passwd} = $h->{-params}->{-new_passwd};
                return $h->committee_menu('PASSWD_CHANGED');
            } else {
                # Whops .. the current password is bad :(
                $template->param(BAD_CURRENT_PASSWD => 1);
                return $template->output;
            }
        }
    } else {
        # If the form hasn't been posted
        return $template->output;
    }
}

sub _ck_committee_task {
    my $h = shift;
    my $admin = Comas::Admin->new(-db => $h->{-db_adm},
                                  %{$h->{-session}->{-admin}});

    if ($admin->ck_admin_task(-task=>$h->{-path}->[1],
                              -login=>$admin->{-login})) {
        return 1;
    } else {
        return 0;
    }
}

1;

# $Log: committee.pm,v $
# Revision 1.5  2003/12/20 04:14:51  mig
# - Agrego tags Id y Log que expanda el CVS
#
